// card.c

// 3-10: Now thread-safe(r) with help from Jon Watte (hplus@b500.com)

#include <KernelExport.h>
#include <Drivers.h>
#include <Errors.h>
#include <string.h>
#include <stdio.h>
#include "AztechRadio.h"

__declspec(dllexport)
// Implemented to comply with Be driver interface; they don't do much
status_t init_hardware(void);	// Mutes card
status_t init_driver(void);
void uninit_driver(void);

//status_t r_open( const char *, uint32, void** );

const char ** publish_devices(void);
device_hooks * find_device(const char *);

///////////////////////////////////////////////
// Read from card
unsigned char read_card(int sub_port);
// Write to card
void write_card(int sub_port, unsigned char value);
// I/O Helper functions

#define DRIVER_DEBUG
#ifdef DRIVER_DEBUG
	#define DRIVER_PRINT(ARG) 			dprintf ARG
#else
	#define DRIVER_PRINT(ARGS)			(void)0
#endif

static int radio_wait_time = 1000;
sem_id g_sem;

struct az_device
{
	int curvol;
	unsigned long curfreq;
	int stereo;
};

static struct az_device aztech_unit;

static int volconvert(int level)
{
	level>>=14;	 	/* Map 16bits down to 2 bit */
 	level&=3;
	
	/* convert to card-friendly values */
	switch (level) 
	{
		case 0: 
			return 0;
		case 1: 
			return 1;
		case 2:
			return 4;
		case 3:
			return 5;
	}
	return 0;	/* Quieten gcc */
}

static void send_0_byte (struct az_device *dev)
{
	spin(radio_wait_time);
	write_card(0, 2+volconvert(dev->curvol));
//	outb_p(2+volconvert(dev->curvol), io);
	write_card(0,64+2+volconvert(dev->curvol));
//	outb_p(64+2+volconvert(dev->curvol), io);
}

static void send_1_byte (struct az_device *dev)
{
	spin (radio_wait_time);
	write_card(0,128+2+volconvert(dev->curvol));
//	outb_p(128+2+volconvert(dev->curvol), io);
	write_card(0,128+64+2+volconvert(dev->curvol));
//	outb_p(128+64+2+volconvert(dev->curvol), io);
}

static int az_setvol(struct az_device *dev, int vol)
{
//	outb (volconvert(vol), io);
	write_card(0,vol);
	return 0;
}

static int az_getsigstr(struct az_device *dev)
{
	if (read_card(0) & 2)	/* bit set = no signal present */
//	if (inb(io) & 2)	/* bit set = no signal present */
		return 0;
	return 1;		/* signal present */
}

static int az_getstereo(struct az_device *dev)
{
	if (read_card(0) & 1)	/* bit set = no signal present */
//	if (inb(io) & 1) 	/* bit set = mono */
		return 0;
	return 1;		/* stereo */
}

static int az_setfreq(struct az_device *dev, unsigned long frequency)
{
	int  i;

	frequency += 171200;		/* Add 10.7 MHz IF		*/
	frequency /= 800;		/* Convert to 50 kHz units	*/
					
	send_0_byte (dev);		/*  0: LSB of frequency       */

	for (i = 0; i < 13; i++)	/*   : frequency bits (1-13)  */
		if (frequency & (1 << i))
			send_1_byte (dev);
		else
			send_0_byte (dev);

	send_0_byte (dev);		/* 14: test bit - always 0    */
	send_0_byte (dev);		/* 15: test bit - always 0    */
	send_0_byte (dev);		/* 16: band data 0 - always 0 */
	if (dev->stereo)		/* 17: stereo (1 to enable)   */
		send_1_byte (dev);
	else
		send_0_byte (dev);

	send_1_byte (dev);		/* 18: band data 1 - unknown  */
	send_0_byte (dev);		/* 19: time base - always 0   */
	send_0_byte (dev);		/* 20: spacing (0 = 25 kHz)   */
	send_1_byte (dev);		/* 21: spacing (1 = 25 kHz)   */
	send_0_byte (dev);		/* 22: spacing (0 = 25 kHz)   */
	send_1_byte (dev);		/* 23: AM/FM (FM = 1, always) */

	/* latch frequency */

	spin (radio_wait_time);
//	outb_p(128+64+volconvert(dev->curvol), io);
	write_card(0,128+64+volconvert(dev->curvol));

	return 0;
}

/* init_hardware - called once the first time the driver is loaded */
status_t
init_hardware (void)
{
	DRIVER_PRINT(("init_hardware\n"));
	return B_OK;
}

status_t
init_driver (void)
{
	DRIVER_PRINT(("init_driver\n"));
	g_sem = create_sem( 1, "AztechRadio lock" );
	
	aztech_unit.stereo = 1;
	aztech_unit.curvol = 30000;
	
	return (g_sem < 0) ? g_sem : B_OK;
}

void
uninit_driver (void)
{
	DRIVER_PRINT(("uninit_driver\n"));
	delete_sem( g_sem );
}

static status_t
r_open (const char *name, uint32 flags, void** cookie)
{
	DRIVER_PRINT(("r_open\n"));
	return B_OK;
}

static status_t
r_read (void* cookie, off_t position, void *buf, size_t* num_bytes)
{
	DRIVER_PRINT(("r_read\n"));
	*num_bytes = 0;				/* tell caller nothing was read */
	return B_IO_ERROR;
}

/* r_write: handle write() calls.
	3/10/99: Functionality temporarily removed because it didn't work right */
static status_t
r_write (void* cookie, off_t position, const void* buffer, size_t* num_bytes)
{
	DRIVER_PRINT(("r_write\n"));
	return B_OK;
}

/* r_control: decides whether to tune or mute the card */
static status_t r_control (void* cookie, uint32 op, void* arg, size_t len)
{
	unsigned long *pt_val = (unsigned long*)arg;
	unsigned long val = 0;
	if (pt_val)
		val = *(unsigned long*)arg;
	DRIVER_PRINT(("r_control\n"));

	switch(op)
	{
		case RADIO_TUNE:
			az_setfreq(&aztech_unit,val);
			break;

		case RADIO_ON:
//			aztech_unmute(&aztech_unit);
			break;
			
		case RADIO_OFF:
//			aztech_mute(&aztech_unit);
			break;
			
		case IS_SIGNAL:
//			aztech_getsigstr(&aztech_unit);
			break;

		case IS_RADIO_MUTE:
			break;
			
		default:
			return EINVAL;
			break;
	}
	return B_OK;
}

static status_t
r_close (void* cookie)
{
	DRIVER_PRINT(("r_close\n"));
	return B_OK;
}

static status_t
r_free (void* cookie)
{
	DRIVER_PRINT(("r_free\n"));
	return B_OK;
}

/* null-terminated array of device names */
static const char *my_device_name[] = {
	"misc/AztechRadio",
	NULL
};

device_hooks my_device_hooks = {
	r_open, 			/* -> open entry point */
	r_close, 			/* -> close entry point */
	r_free,				/* -> free cookie */
	r_control, 			/* -> control entry point */
	r_read,				/* -> read entry point */
	r_write				/* -> write entry point */
};

/* publish_devices - return a null-terminated array of devices
	supported by this driver */
const char**
publish_devices()
{
	return my_device_name;
}

/* find_device - return ptr to device hooks structure */
device_hooks*
find_device(const char* name)
{
	return &my_device_hooks;
}

/* reads a byte in from the card */
unsigned char read_card(int sub_port)
{
   return read_io_8(RADIO_BASE+sub_port);
}

/* writes a byte to the card */
void write_card(int sub_port, unsigned char value)
{
   write_io_8(RADIO_BASE+sub_port,value);
}
